--
--****************************
--*** Minimal Display Bars ***
--****************************
--* Coded by: ATPHHe
--* Date Created: 02/19/2020
--* Date Modified: 02/29/2020
--*******************************
--
--============================================================
local MOD_ID = "MinimalHPBar"

local gameVersion = getCore():getVersionNumber()
local gameVersionNum = 0
local tempIndex, _ = string.find(gameVersion, " ")
if tempIndex ~= nil then
    
    gameVersionNum = tonumber(string.sub(gameVersion, 0, tempIndex))
    if gameVersionNum == nil then 
        tempIndex, _ = string.find(gameVersion, ".") + 1 
        gameVersionNum = tonumber(string.sub(gameVersion, 0, tempIndex))
    end
else
    gameVersionNum = tonumber(gameVersion)
end
tempIndex = nil
gameVersion = nil

local defaultSettingsFileName = "MOD DefaultSettings (".. MOD_ID ..").lua"
local configFileName = "MOD Config Options (".. MOD_ID ..").lua"
--local configFileLocation = getMyDocumentFolder() .. getFileSeparator() .. configFileName
local configFileLocation = configFileName

--local configOpts = {}
--[[table.insert(configOpts, "x")
table.insert(configOpts, "y")
table.insert(configOpts, "isVertical")
configOpts["x"] = 70
configOpts["y"] = 30
configOpts["isVertical"] = 1]]

--============================================================

--*********************************************
-- Other Useful Functions

local function tprint(tbl, indent)
    if not indent then indent = 0 end
    for k, v in pairs(tbl) do
        formatting = string.rep("  ", indent) .. k .. ": "
        if type(v) == "table" then
            print(formatting)
            tprint(v, indent+1)
        else
            print(formatting .. v)
        end
    end
end

local function deepcompare(t1,t2,ignore_mt)
    local ty1 = type(t1)
    local ty2 = type(t2)
    if ty1 ~= ty2 then return false end
    -- non-table types can be directly compared
    if ty1 ~= 'table' and ty2 ~= 'table' then return t1 == t2 end
    -- as well as tables which have the metamethod __eq
    local mt = getmetatable(t1)
    if not ignore_mt and mt and mt.__eq then return t1 == t2 end
    for k1,v1 in pairs(t1) do
        local v2 = t2[k1]
        if v2 == nil or not deepcompare(v1,v2) then return false end
    end
    for k2,v2 in pairs(t2) do
        local v1 = t1[k2]
        if v1 == nil or not deepcompare(v1,v2) then return false end
    end
    return true
end

local function compare_and_insert(t1, t2, ignore_mt)
    
    local isEqual = true
    
    if not t1 then
        return false
    end
    
    if not t2 then
        t2 = {}
        isEqual = false
    end
    
    for k1,v1 in pairs(t1) do
        local v2 = t2[k1]
        if (v2 == nil) then 
            t2[k1] = v1
            isEqual = false
        end
        
        if type(t1[k1]) == "table" then
            isEqual = compare_and_insert(t1[k1], t2[k1], ignore_mt)
        end
        
    end
    
    return isEqual
end

-- Splits the string apart.
--  EX: inputstr = "Hello There Friend."
--      sep = " "
--      t = {Hello, 
--          There, 
--          Friend.}
--  EX: inputstr = "Hello,There,Friend."
--      sep = ","
--      t = {Hello, 
--          There, 
--          Friend.}
--
-- Parameters:  inputstr - the string that will be split.
--              sep - the separator character that will be used to split the string
--              t - the table that will be returned.
--
local function split(inputstr, sep)
    if sep == nil then
        sep = "%s"
    end
    local t={}
    for str in string.gmatch(inputstr, "([^"..sep.."]+)") do
        table.insert(t, str)
    end
    return t
end

-- Returns true, if "a" a number. Otherwise return false.
local function isNumber(a)
    if tonumber(a) ~= nil then
        local number = tonumber(a)
        if number then
            return true
        end
    end
    
    return false
end


--**********************************************
-- Vanilla Functions


--*************************
-- I/O Functions


--[[

TablePersistence is a small code snippet that allows storing and loading of lua variables containing primitive types. It is licensed under the MIT license, use it how ever is needed. A more detailed description and complete source can be downloaded on http://the-color-black.net/blog/article/LuaTablePersistence. A fork has been created on github that included lunatest unit tests: https://github.com/hipe/lua-table-persistence

Shortcomings/Limitations:
- Does not export udata
- Does not export threads
- Only exports a small subset of functions (pure lua without upvalue)

]]
local write, writeIndent, writers, refCount;
local io_persistence =
{
	store = function (path, modID, ...)
		local file, e = getModFileWriter(modID, path, true, false) --e = io.open(path, "w");
		if not file then
			return error(e);
		end
		local n = select("#", ...);
		-- Count references
		local objRefCount = {}; -- Stores reference that will be exported
		for i = 1, n do
			refCount(objRefCount, (select(i,...)));
		end;
		-- Export Objects with more than one ref and assign name
		-- First, create empty tables for each
		local objRefNames = {};
		local objRefIdx = 0;
		file:write("-- Persistent Data (for "..modID..")\n");
		file:write("local multiRefObjects = {\n");
		for obj, count in pairs(objRefCount) do
			if count > 1 then
				objRefIdx = objRefIdx + 1;
				objRefNames[obj] = objRefIdx;
				file:write("{};"); -- table objRefIdx
			end;
		end;
		file:write("\n} -- multiRefObjects\n");
		-- Then fill them (this requires all empty multiRefObjects to exist)
		for obj, idx in pairs(objRefNames) do
			for k, v in pairs(obj) do
				file:write("multiRefObjects["..idx.."][");
				write(file, k, 0, objRefNames);
				file:write("] = ");
				write(file, v, 0, objRefNames);
				file:write(";\n");
			end;
		end;
		-- Create the remaining objects
		for i = 1, n do
			file:write("local ".."obj"..i.." = ");
			write(file, (select(i,...)), 0, objRefNames);
			file:write("\n");
		end
		-- Return them
		if n > 0 then
			file:write("return obj1");
			for i = 2, n do
				file:write(" ,obj"..i);
			end;
			file:write("\n");
		else
			file:write("return\n");
		end;
		if type(path) == "string" then
			file:close();
		end;
	end;

	load = function (path, modID)
		local f, e;
		if type(path) == "string" then
            --f, e = loadfile(path);
			f, e = getModFileReader(modID, path, true);
            if f == nil then f = getFileReader(sourceFile, true) end;
            
            local contents = "";
            local scanLine = f:readLine();
            while scanLine do
                
                contents = contents.. scanLine .."\r\n";
                
                scanLine = f:readLine();
                if not scanLine then break end
            end
            
            f:close();
            
            f = contents;
		else
			f, e = path:read('*a');
		end
		if f then
            local func = loadstring(f);
            if func then
                return func();
            else
                return nil;
            end
		else
			return nil, e;
		end;
	end;
}

-- Private methods

-- write thing (dispatcher)
write = function (file, item, level, objRefNames)
	writers[type(item)](file, item, level, objRefNames);
end;

-- write indent
writeIndent = function (file, level)
	for i = 1, level do
		file:write("\t");
	end;
end;

-- recursively count references
refCount = function (objRefCount, item)
	-- only count reference types (tables)
	if type(item) == "table" then
		-- Increase ref count
		if objRefCount[item] then
			objRefCount[item] = objRefCount[item] + 1;
		else
			objRefCount[item] = 1;
			-- If first encounter, traverse
			for k, v in pairs(item) do
				refCount(objRefCount, k);
				refCount(objRefCount, v);
			end;
		end;
	end;
end;

-- Format items for the purpose of restoring
writers = {
	["nil"] = function (file, item)
			file:write("nil");
		end;
	["number"] = function (file, item)
			file:write(tostring(item));
		end;
	["string"] = function (file, item)
			file:write(string.format("%q", item));
		end;
	["boolean"] = function (file, item)
			if item then
				file:write("true");
			else
				file:write("false");
			end
		end;
	["table"] = function (file, item, level, objRefNames)
			local refIdx = objRefNames[item];
			if refIdx then
				-- Table with multiple references
				file:write("multiRefObjects["..refIdx.."]");
			else
				-- Single use table
				file:write("{\r\n");
				for k, v in pairs(item) do
					writeIndent(file, level+1);
					file:write("[");
					write(file, k, level+1, objRefNames);
					file:write("] = ");
					write(file, v, level+1, objRefNames);
					file:write(";\r\n");
				end
				writeIndent(file, level);
				file:write("}");
			end;
		end;
	["function"] = function (file, item)
			-- Does only work for "normal" functions, not those
			-- with upvalues or c functions
			local dInfo = debug.getinfo(item, "uS");
			if dInfo.nups > 0 then
				file:write("nil --[[functions with upvalue not supported]]");
			elseif dInfo.what ~= "Lua" then
				file:write("nil --[[non-lua function not supported]]");
			else
				local r, s = pcall(string.dump,item);
				if r then
					file:write(string.format("loadstring(%q)", s));
				else
					file:write("nil --[[function could not be dumped]]");
				end
			end
		end;
	["thread"] = function (file, item)
			file:write("nil --[[thread]]\r\n");
		end;
	["userdata"] = function (file, item)
			file:write("nil --[[userdata]]\r\n");
		end;
}

-- Testing Persistence
--io_persistence.store("storage.lua", MOD_ID, configOpts)
--t_restored = io_persistence.load("storage.lua", MOD_ID);
--io_persistence.store("storage2.lua", MOD_ID, t_restored)



-- Save to a destination file.
-- Returns true if successful, otherwise return false if an error occured.
local function SaveToFile(destinationFile, text)
    local fileWriter = getModFileWriter(MOD_ID, destinationFile, true, false)
    if fileWriter == nil then fileWriter = getFileWriter(destinationFile, true, false) end
    
    fileWriter:write(tostring(text))
    fileWriter:close()
end

-- Load from a sourceFile file.
-- Returns a table of Strings, representing each line in the file.
local function LoadFromFile(sourceFile)

	local contents = {}
	local fileReader = getModFileReader(MOD_ID, sourceFile, true)
    if fileReader == nil then fileReader = getFileReader(sourceFile, true) end
    
	local scanLine = fileReader:readLine()
	while scanLine do
        
        table.insert(contents, tostring(scanLine))
        
		scanLine = fileReader:readLine()
		if not scanLine then break end
	end
    
	fileReader:close();
    
	return contents
end

-- Recreates the default configuration files for this mod.
local function recreateConfigFiles()
    local fileContents1 = io_persistence.load(defaultSettingsFileName, MOD_ID)
    io_persistence.store(configFileLocation, MOD_ID, fileContents1)
    return fileContents1
end


--*********************************************
-- Custom Tables
local MinimalDisplayBarsDefaultSettings = {
    
    ["lock_all_bars"] = false,
    
    ["menu"] = {
        ["x"] = 70,
        ["y"] = 15,
        ["width"] = 15,
        ["height"] = 15,
        ["l"] = 3,
        ["t"] = 3,
        ["r"] = 3,
        ["b"] = 3,
        ["color"] = {red = (255 / 255), 
                    green = (255 / 255), 
                    blue = (255 / 255), 
                    alpha = 0.75},
        ["isMovable"] = true,
        ["isResizable"] = false,
        ["isVisible"] = true,
        ["isVertical"] = true,
        ["alwaysBringToTop"] = true,
        },
    ["hp"] = {
        ["x"] = 70,
        ["y"] = 30,
        ["width"] = 15,
        ["height"] = 150,
        ["l"] = 3,
        ["t"] = 3,
        ["r"] = 3,
        ["b"] = 3,
        ["color"] = {red = (0 / 255), 
                    green = (128 / 255), 
                    blue = (0 / 255), 
                    alpha = 0.75},
        ["isMovable"] = true,
        ["isResizable"] = false,
        ["isVisible"] = true,
        ["isVertical"] = true,
        ["alwaysBringToTop"] = true,
        },
    ["hunger"] = {
        ["x"] = 85,
        ["y"] = 30,
        ["width"] = 8,
        ["height"] = 150,
        ["l"] = 2,
        ["t"] = 3,
        ["r"] = 2,
        ["b"] = 3,
        ["color"] = {red = (255 / 255), 
                    green = (255 / 255), 
                    blue = (10 / 255), 
                    alpha = 0.75},
        ["isMovable"] = true,
        ["isResizable"] = false,
        ["isVisible"] = true,
        ["isVertical"] = true,
        ["alwaysBringToTop"] = true,
        },
    ["thirst"] = {
        ["x"] = 93,
        ["y"] = 30,
        ["width"] = 8,
        ["height"] = 150,
        ["l"] = 2,
        ["t"] = 3,
        ["r"] = 2,
        ["b"] = 3,
        ["color"] = {red = (173 / 255), 
                    green = (216 / 255), 
                    blue = (230 / 255), 
                    alpha = 0.75},
        ["isMovable"] = true,
        ["isResizable"] = false,
        ["isVisible"] = true,
        ["isVertical"] = true,
        ["alwaysBringToTop"] = true,
        },
    ["endurance"] = {
        ["x"] = 101,
        ["y"] = 30,
        ["width"] = 8,
        ["height"] = 150,
        ["l"] = 2,
        ["t"] = 3,
        ["r"] = 2,
        ["b"] = 3,
        ["color"] = {red = (244 / 255), 
                    green = (244 / 255), 
                    blue = (244 / 255), 
                    alpha = 0.75},
        ["isMovable"] = true,
        ["isResizable"] = false,
        ["isVisible"] = true,
        ["isVertical"] = true,
        ["alwaysBringToTop"] = true,
        },
    ["fatigue"] = {
        ["x"] = 109,
        ["y"] = 30,
        ["width"] = 8,
        ["height"] = 150,
        ["l"] = 2,
        ["t"] = 3,
        ["r"] = 2,
        ["b"] = 3,
        ["color"] = {red = (240 / 255), 
                    green = (240 / 255), 
                    blue = (170 / 255), 
                    alpha = 0.75},
        ["isMovable"] = true,
        ["isResizable"] = false,
        ["isVisible"] = true,
        ["isVertical"] = true,
        ["alwaysBringToTop"] = true,
        },
    }

local configOpts = io_persistence.load(defaultSettingsFileName, MOD_ID)
if not deepcompare(configOpts, MinimalDisplayBarsDefaultSettings, false) then
    io_persistence.store(defaultSettingsFileName, MOD_ID, MinimalDisplayBarsDefaultSettings)
    configOpts = io_persistence.load(defaultSettingsFileName, MOD_ID)
end
--===============================================================================
-- Get/Setup all configuration settings from the config file.
local t_restored = io_persistence.load(configFileLocation, MOD_ID)

if not compare_and_insert(configOpts, t_restored, true) then
    io_persistence.store(configFileLocation, MOD_ID, t_restored)
end

if t_restored then 
    configOpts = t_restored 
else 
    configOpts = recreateConfigFiles()
end

--*************************************
-- Custom Functions
local barMenu
local barHP
local barHunger
local barThirst
local barEndurance
local barFatigue

-- ContextMenu
local function showContextMenu(generic_bar, dx, dy)
	local contextMenu = ISContextMenu.get(
                            generic_bar.playerIndex, 
                            (generic_bar.x + dx), (generic_bar.y + dy), 1, 1)
    -- Title
	contextMenu:addOption("--- " .. getText("ContextMenu_MinimalHPBar_Title") .. " ---")
    
    -- Menu
    if generic_bar.idName == "menu" then
        contextMenu:addOption(getText("ContextMenu_MinimalHPBar_Reset"),
                    generic_bar,
                    function(generic_bar)
                    
                        configOpts = io_persistence.load(defaultSettingsFileName, MOD_ID)
                        io_persistence.store(configFileLocation, MOD_ID, configOpts)
                        
                        if generic_bar then 
                            generic_bar:resetBar() end
                        if barHP then 
                            barHP:resetBar() end
                        if barHunger then 
                            barHunger:resetBar() end
                        if barThirst then 
                            barThirst:resetBar() end
                        if barEndurance then 
                            barEndurance:resetBar() end
                        if barFatigue then 
                            barFatigue:resetBar() end
                        
                        return
                    end)
        contextMenu:addOption(getText("ContextMenu_MinimalHPBar_Make_All_Bars_Visible"),
                    generic_bar,
                    function(generic_bar)
                    
                        configOpts = io_persistence.load(defaultSettingsFileName, MOD_ID)
                        io_persistence.store(configFileLocation, MOD_ID, configOpts)
                        
                        if generic_bar then 
                            generic_bar:setVisible(true) end
                        if barHP then 
                            barHP:setVisible(true) end
                        if barHunger then 
                            barHunger:setVisible(true) end
                        if barThirst then 
                            barThirst:setVisible(true) end
                        if barEndurance then 
                            barEndurance:setVisible(true) end
                        if barFatigue then 
                            barFatigue:setVisible(true) end
                        
                        return
                    end)
        contextMenu:addOption(
                    getText("ContextMenu_MinimalHPBar_Toggle_Always_Bring_Display_Bars_To_Top"),
                    generic_bar,
                    function(generic_bar)
                        local function toggleAlwaysBringToTop(bar)
                            if bar.alwaysBringToTop then
                                bar.alwaysBringToTop = false
                                configOpts[bar.idName]["alwaysBringToTop"] = false
                            else
                                bar.alwaysBringToTop = true
                                configOpts[bar.idName]["alwaysBringToTop"] = true
                            end
                        end
                        
                        if barHP then 
                            toggleAlwaysBringToTop(barHP) end
                        if barHunger then 
                            toggleAlwaysBringToTop(barHunger) end
                        if barThirst then 
                            toggleAlwaysBringToTop(barThirst) end
                        if barEndurance then 
                            toggleAlwaysBringToTop(barEndurance) end
                        if barFatigue then 
                            toggleAlwaysBringToTop(barFatigue) end
                        
                        io_persistence.store(configFileLocation, MOD_ID, configOpts)
                        
                        return
                    end)
    else
    
    -- Display Bars
        contextMenu:addOption(getText("ContextMenu_MinimalHPBar_Set_Vertical"),
                    generic_bar,
                    function(generic_bar)
                        
                        if not generic_bar then return end
                        
                        if configOpts[generic_bar.idName]["isVertical"] == false then 
                            generic_bar.isVertical = true
                            configOpts[generic_bar.idName]["isVertical"] = true
                            
                            local oldW = tonumber(generic_bar.oldWidth)
                            local oldH = tonumber(generic_bar.oldHeight)
                            generic_bar:setWidth(oldH)
                            generic_bar:setHeight(oldW)
                            
                            io_persistence.store(configFileLocation, MOD_ID, configOpts)
                        end
                        return
                    end)
        contextMenu:addOption(getText("ContextMenu_MinimalHPBar_Set_Horizontal"),
                    generic_bar,
                    function(generic_bar)
                    
                        if not generic_bar then return end
                        
                        if configOpts[generic_bar.idName]["isVertical"] == true then 
                            generic_bar.isVertical = false
                            configOpts[generic_bar.idName]["isVertical"] = false
                            
                            local oldW = tonumber(generic_bar.oldWidth)
                            local oldH = tonumber(generic_bar.oldHeight)
                            generic_bar:setWidth(oldH)
                            generic_bar:setHeight(oldW)
                            
                            io_persistence.store(configFileLocation, MOD_ID, configOpts)
                        end
                        return
                    end)
        contextMenu:addOption(getText("ContextMenu_MinimalHPBar_Hide_Bar"),
                    generic_bar,
                    function(generic_bar)
                    
                        if not generic_bar then return end
                        generic_bar:setVisible(false)
                        
                        return
                    end)
    end
    
end

-- GenericMiniDisplayBar
local GenericMiniDisplayBar = ISPanel:derive("GenericMiniDisplayBar")

function GenericMiniDisplayBar:setWidth(w, ...)
    local panel = ISPanel.setWidth(self, w, ...)
    self.oldWidth = self.width
    self.innerWidth = (self.width - self.borderSizes.l - self.borderSizes.r)
    return panel
end

function GenericMiniDisplayBar:setHeight(h, ...)
    local panel = ISPanel.setHeight(self, h, ...)
    self.oldHeight = self.height
    self.innerHeight = (self.height - self.borderSizes.t - self.borderSizes.b)
    return panel
end

function GenericMiniDisplayBar:onMouseDoubleClick(x, y, ...)
    return
end

function GenericMiniDisplayBar:onRightMouseDown(x, y, ...)
    local result = ISPanel.onRightMouseDown(self, x, y, ...)
    self.rightMouseDown = true
    return result
end

function GenericMiniDisplayBar:onRightMouseUp(dx, dy, ...)
    local panel = ISPanel.onRightMouseUp(self, dx, dy, ...)
    if self.rightMouseDown == true then showContextMenu(self, dx, dy) end
	self.rightMouseDown = false
    return panel
end

function GenericMiniDisplayBar:onRightMouseUpOutside(x, y, ...)
	local panel = ISPanel.onRightMouseUpOutside(self, x, y, ...)
	self.rightMouseDown = false
	return panel
end

function GenericMiniDisplayBar:onMouseDown(x, y, ...)
    local panel = ISPanel.onMouseDown(self, x, y, ...)
    self.oldX = self.x
    self.oldY = self.y
    return panel
end

function GenericMiniDisplayBar:render(...)
    local panel = ISPanel.render(self, ...)
    
    local innerWidth = 0
    local innerHeight = 0
    local border_t = 0
    
    -- Make sure the bar stays on screen when the window is resized
    if getPlayerScreenWidth(self.playerIndex) < self:getX() + self:getWidth() then
        self:setX(self:getX() - self:getWidth())
    end
    
    if getPlayerScreenHeight(self.playerIndex) < self:getY() + self:getHeight() then
        self:setY(self:getY() - self:getHeight())
    end
    
    -- Use the color function of this bar if one exists.
    local value = self.valueFunction.getValue(self.isoPlayer)
    local colorFunc = self.colorFunction
    local color 
    if colorFunc ~= nil and colorFunc.getColor ~= nil then
        color = colorFunc.getColor(self.isoPlayer)
        if color ~= nil and self.useColorFunction == true then self.color = color end
    end
    
    -- Automatically picks the way that the bar will decrease and increase visually.
    if self.width > self.height then
         innerWidth = math.floor((self.innerWidth * value) + 0.5)
         innerHeight = self.innerHeight
         border_t = self.borderSizes.t
    else 
         innerWidth = self.innerWidth
         innerHeight = math.floor((self.innerHeight * value) + 0.5)
         border_t = self.borderSizes.t + ((self.height - self.borderSizes.t - self.borderSizes.b) - innerHeight)
    end
    
    -- if value is less than or equal to -1, make this bar not visible
    if value <= -1 then 
        if self:isVisible() then self:setVisible(false) end
    else 
        if not self:isVisible() then self:setVisible(true) end
    end
    
    --[[self:drawRectStatic(
        self.borderSizes.l,
        self.borderSizes.t,
        self.innerWidth,
        self.innerHeight,
        self.color.alpha,
        0.3,
        0.05,
        0.05)]]
    
    self:drawRectStatic(
        self.borderSizes.l,
        border_t,
        innerWidth,
        innerHeight,
        self.color.alpha,
        self.color.red,
        self.color.green,
        self.color.blue)
            
	if self.moving or self.resizing then
		self:drawRectStatic(
            self.borderSizes.l,
            border_t,
            innerWidth,
            innerHeight,
            0.5,
            0,
            0,
            0)
        self:drawText(
            ("x: "..self.x.." \r\ny: "..self.y),
            self.borderSizes.l,
            self.borderSizes.t,
            1,
            1,
            1,
            1,
            UIFont.Small)
    end

    -- Text Debugger for stats.
    --[[self:drawText(
        ("Hunger: "..(math.pow(1 - self.isoPlayer:getStats():getHunger(), 2) ).."\r\n"..
            "Thirst: "..(math.pow(1 - self.isoPlayer:getStats():getThirst(), 2) ).."\r\n"..
            "Fatigue: "..(math.pow(1 - self.isoPlayer:getStats():getFatigue(), 2) ).."\r\n"..
            "Endurance: "..((self.isoPlayer:getStats():getEndurance()) ).."\r\n"..
            ""
        ),
        self.borderSizes.l,
        self.borderSizes.t,
        1,
        1,
        1,
        1,
        UIFont.Small)]]

    -- Save to file if the bar was suddenly moved.
    if self.moving == false then
        if self.oldX ~= self.x or self.oldY ~= self.y then
            self.oldX = self.x
            self.oldY = self.y
            configOpts[self.idName]["x"] = self.x
            configOpts[self.idName]["y"] = self.y
            io_persistence.store(configFileLocation, MOD_ID, configOpts)
        end
    end
    
    -- force to the top of UI if true.
    if self.alwaysBringToTop == true then self:bringToTop() end
    
    return panel
end

function GenericMiniDisplayBar:resetBar(...)
    --local panel = ISPanel.resetBar(self, ...)
    
    self.x = (configOpts[self.idName]["x"])
    self.y = (configOpts[self.idName]["y"])
    self.oldX = self.x
    self.oldY = self.y
    
    self:setWidth(configOpts[self.idName]["width"])
    self:setHeight(configOpts[self.idName]["height"])
    
    self.oldWidth = self.width
    self.oldHeight = self.height
    
	self.moveWithMouse = configOpts[self.idName]["isMovable"]
	self.resizeWithMouse = configOpts[self.idName]["isResizable"]
    
	self.borderSizes = {l = configOpts[self.idName]["l"], 
                        t = configOpts[self.idName]["t"], 
                        r = configOpts[self.idName]["r"], 
                        b = configOpts[self.idName]["b"]}
	self.innerWidth = (self.width - self.borderSizes.l - self.borderSizes.r)
	self.innerHeight = (self.height - self.borderSizes.t - self.borderSizes.b)
	self.color = configOpts[self.idName]["color"]
	self.minimumWidth = (1 + self.borderSizes.l + self.borderSizes.r)
	self.minimumHeight = (1 + self.borderSizes.t + self.borderSizes.b)
    
    self.isVertical = configOpts[self.idName]["isVertical"]
    
	self:setVisible(configOpts[self.idName]["isVisible"])
    
    self.alwaysBringToTop = configOpts[self.idName]["alwaysBringToTop"]
    self:setAlwaysOnTop(false)
    --return panel
end

function GenericMiniDisplayBar:new(
                                idName,
                                playerIndex, isoPlayer, 
                                x, y, 
                                width, height, 
                                l, t, r, b,
                                color, 
                                isVisible, 
                                isMovable, 
                                isResizable, 
                                isVertical,
                                alwaysBringToTop,
                                bChild, 
                                valueFunction, 
                                colorFunction, useColorFunction)
	local panel = ISPanel:new(x, y, width, height)
	setmetatable(panel, self)
	self.__index = self
    
    panel.idName = idName
    
    panel.oldX = panel.x
    panel.oldY = panel.y
    panel.oldWidth = panel.width
    panel.oldHeight = panel.height
    
	panel.playerIndex = playerIndex
    panel.isoPlayer = isoPlayer
    
	panel.moveWithMouse = isMovable
	panel.resizeWithMouse = isResizable
    
	panel.borderSizes = {l = l, t = t, r = r, b = b}
	panel.innerWidth = (panel.width - panel.borderSizes.l - panel.borderSizes.r)
	panel.innerHeight = (panel.height - panel.borderSizes.t - panel.borderSizes.b)
	panel.color = color
	panel.minimumWidth = (1 + panel.borderSizes.l + panel.borderSizes.r)
	panel.minimumHeight = (1 + panel.borderSizes.t + panel.borderSizes.b)
    
	panel.valueFunction = {getValue = valueFunction}
    panel.colorFunction = {getColor = colorFunction}
    panel.useColorFunction = useColorFunction
    panel.isVertical = isVertical
    
    panel.bChild = bChild
    
    --panel.lock = false
    
    if panel.isVertical then
        if panel.width > panel.height then
            local oldW = tonumber(panel.oldWidth)
            local oldH = tonumber(panel.oldHeight)
            panel:setWidth(oldH)
            panel:setHeight(oldW)
        end
    else
        if panel.width < panel.height then
            local oldW = tonumber(panel.oldWidth)
            local oldH = tonumber(panel.oldHeight)
            panel:setWidth(oldH)
            panel:setHeight(oldW)
        end
    end
    
	panel:setVisible(isVisible)
    
    self.alwaysBringToTop = alwaysBringToTop
    panel:setAlwaysOnTop(false)
	return panel
end

-- Health Functions
local function getHealth(isoPlayer) 
    if isoPlayer:isDead() then
        return -1
    else
        return isoPlayer:getBodyDamage():getHealth() / 100 
    end
end

local function getColorHP(isoPlayer) 
    local hpRatio = 0
    if not isoPlayer:isDead() then
        hpRatio = isoPlayer:getBodyDamage():getHealth() / 100 
    end
    
    local color
    if hpRatio < 1 then
        color = { red = (255 / 255), 
                    green = (255 / 255) * (math.pow(0.1, 1 - hpRatio)), 
                    blue = (10 / 255) * (1 - hpRatio), 
                    alpha = 0.75 }
    else 
        color = { red = (255 / 255) * (1 - hpRatio), 
                    green = (128 / 255) * (hpRatio), 
                    blue = (10 / 255) * (1 - hpRatio), 
                    alpha = 0.75 }
    end
    return color
end

-- Hunger Functions
local function getHunger(isoPlayer) 
    if isoPlayer:isDead() then
        return -1
    else
        return (math.pow(1 - isoPlayer:getStats():getHunger(), 2) )
    end
end

local function getColorHunger(isoPlayer) 
    local color
    color = configOpts["hunger"]["color"]
    return color
end

-- Thirst Functions
local function getThirst(isoPlayer) 
    if isoPlayer:isDead() then
        return -1
    else
        return (math.pow(1 - isoPlayer:getStats():getThirst(), 2) )
    end
end

local function getColorThirst(isoPlayer) 
    local color
    color = configOpts["thirst"]["color"]
    return color
end

-- Endurance Functions
local function getEndurance(isoPlayer) 
    if isoPlayer:isDead() then
        return -1
    else
        return ((isoPlayer:getStats():getEndurance()) )
    end
end

local function getColorEndurance(isoPlayer) 
    local color
    color = configOpts["endurance"]["color"]
    return color
end

-- Fatigue Functions
local function getFatigue(isoPlayer) 
    if isoPlayer:isDead() then
        return -1
    else
        return ((isoPlayer:getStats():getFatigue()) )
    end
end

local function getColorFatigue(isoPlayer) 
    local color
    color = configOpts["fatigue"]["color"]
    return color
end

-- ...

local function createUI(playerIndex, isoPlayer)

    if not compare_and_insert(configOpts, t_restored, true) then
        io_persistence.store(configFileLocation, MOD_ID, t_restored)
    end

    if t_restored then 
        configOpts = t_restored 
    else 
        configOpts = recreateConfigFiles()
    end
    
    local idName = "menu"
    barMenu = GenericMiniDisplayBar:new(
                    idName,
                    playerIndex, isoPlayer, 
                    configOpts[idName]["x"], configOpts[idName]["y"], 
                    configOpts[idName]["width"], configOpts[idName]["height"], 
                    configOpts[idName]["l"], configOpts[idName]["t"], 
                    configOpts[idName]["r"], configOpts[idName]["b"], 
                    configOpts[idName]["color"], 
                    configOpts[idName]["isVisible"], 
                    configOpts[idName]["isMovable"], 
                    configOpts[idName]["isResizable"], 
                    configOpts[idName]["isVertical"], 
                    configOpts[idName]["alwaysBringToTop"], 
                    nil, 
                    function(tIsoPlayer) 
                        if tIsoPlayer:isDead() then return -1 else return 1 end 
                    end,
                    nil, false)
    barMenu:initialise()
    barMenu:addToUIManager()
    
    local idName = "hp"
    barHP = GenericMiniDisplayBar:new(
                    idName,
                    playerIndex, isoPlayer, 
                    configOpts[idName]["x"], configOpts[idName]["y"], 
                    configOpts[idName]["width"], configOpts[idName]["height"], 
                    configOpts[idName]["l"], configOpts[idName]["t"], 
                    configOpts[idName]["r"], configOpts[idName]["b"], 
                    configOpts[idName]["color"], 
                    configOpts[idName]["isVisible"], 
                    configOpts[idName]["isMovable"], 
                    configOpts[idName]["isResizable"], 
                    configOpts[idName]["isVertical"], 
                    configOpts[idName]["alwaysBringToTop"], 
                    nil, 
                    getHealth,
                    getColorHP, true)
    barHP:initialise()
    barHP:addToUIManager()
    
    --[[local idName = "hunger"
    barHunger = GenericMiniDisplayBar:new(
                    idName,
                    playerIndex, isoPlayer, 
                    configOpts[idName]["x"], configOpts[idName]["y"], 
                    configOpts[idName]["width"], configOpts[idName]["height"], 
                    configOpts[idName]["l"], configOpts[idName]["t"], 
                    configOpts[idName]["r"], configOpts[idName]["b"], 
                    configOpts[idName]["color"], 
                    configOpts[idName]["isVisible"], 
                    configOpts[idName]["isMovable"], 
                    configOpts[idName]["isResizable"], 
                    configOpts[idName]["isVertical"], 
                    configOpts[idName]["alwaysBringToTop"], 
                    nil, 
                    getHunger,
                    getColorHunger, true)
    barHunger:initialise()
    barHunger:addToUIManager()
    
    local idName = "thirst"
    barThirst = GenericMiniDisplayBar:new(
                    idName,
                    playerIndex, isoPlayer, 
                    configOpts[idName]["x"], configOpts[idName]["y"], 
                    configOpts[idName]["width"], configOpts[idName]["height"], 
                    configOpts[idName]["l"], configOpts[idName]["t"], 
                    configOpts[idName]["r"], configOpts[idName]["b"], 
                    configOpts[idName]["color"], 
                    configOpts[idName]["isVisible"], 
                    configOpts[idName]["isMovable"], 
                    configOpts[idName]["isResizable"], 
                    configOpts[idName]["isVertical"], 
                    configOpts[idName]["alwaysBringToTop"], 
                    nil, 
                    getThirst,
                    getColorThirst, true)
    barThirst:initialise()
    barThirst:addToUIManager()
    
    local idName = "endurance"
    barEndurance = GenericMiniDisplayBar:new(
                    idName,
                    playerIndex, isoPlayer, 
                    configOpts[idName]["x"], configOpts[idName]["y"], 
                    configOpts[idName]["width"], configOpts[idName]["height"], 
                    configOpts[idName]["l"], configOpts[idName]["t"], 
                    configOpts[idName]["r"], configOpts[idName]["b"], 
                    configOpts[idName]["color"], 
                    configOpts[idName]["isVisible"], 
                    configOpts[idName]["isMovable"], 
                    configOpts[idName]["isResizable"], 
                    configOpts[idName]["isVertical"], 
                    configOpts[idName]["alwaysBringToTop"], 
                    nil, 
                    getEndurance,
                    getColorEndurance, true)
    barEndurance:initialise()
    barEndurance:addToUIManager()
    
    local idName = "fatigue"
    barFatigue = GenericMiniDisplayBar:new(
                    idName,
                    playerIndex, isoPlayer, 
                    configOpts[idName]["x"], configOpts[idName]["y"], 
                    configOpts[idName]["width"], configOpts[idName]["height"], 
                    configOpts[idName]["l"], configOpts[idName]["t"], 
                    configOpts[idName]["r"], configOpts[idName]["b"], 
                    configOpts[idName]["color"], 
                    configOpts[idName]["isVisible"], 
                    configOpts[idName]["isMovable"], 
                    configOpts[idName]["isResizable"], 
                    configOpts[idName]["isVertical"], 
                    configOpts[idName]["alwaysBringToTop"], 
                    nil, 
                    getFatigue,
                    getColorFatigue, true)
    barFatigue:initialise()
    barFatigue:addToUIManager()]]
end

Events.OnCreatePlayer.Add(createUI)




